<?php

namespace App\Http\Controllers;

use App\Mail\ContactUs;
use App\Models\Blog;
use App\Models\Destination;
use App\Models\Newsletter;
use App\Models\Package;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class FrontendController extends Controller
{
    public function index()
    {
        $destinations = Destination::where('status', true)->limit(6)->get();
        $packages = Package::with(['image', 'destination'])->withCount('reviews')->withAvg('reviews', 'rating')->active()->published()->limit(6)->get();
        $blogs = Blog::with(['category', 'destination'])->where('status', true)->limit(4)->get();

        return view('frontend.index', compact('destinations', 'packages', 'blogs'));
    }

    public function about()
    {
        return view('frontend.pages.about');
    }

    public function destination()
    {
        $destinations = Destination::where('status', true)->paginate(9);

        // return asset($destinations->first()->thumbnail);
        return view('frontend.pages.destination', compact('destinations'));
    }

    public function destinationShow(Destination $destination)
    {
        return view('frontend.pages.destination-show', compact('destination'));
    }

    public function package(Request $request)
    {
        $request->validate([
            'search' => 'nullable|string',
        ]);

        $searchTerm = $request->search;
        $packages = Package::when($searchTerm, function ($query) use ($searchTerm) {
            $query->where(function ($q) use ($searchTerm) {
                $q->whereAny(['name', 'description'], 'like', "%{$searchTerm}%")
                    ->orWhereHas('destination', function ($subQuery) use ($searchTerm) {
                        $subQuery->where('name', 'like', "%{$searchTerm}%");
                    });
            });
        })
            ->active()
            ->published()
            ->with(['image', 'destination'])
            ->withCount('reviews')->withAvg('reviews', 'rating')
            ->paginate(9);

        return view('frontend.pages.package', compact('packages'));
    }

    public function packageShow(Destination $destination, Package $package)
    {
        // return $package->bookings->sum('peoples');
        $availableSeats = $package->max_people - $package->bookings->sum('peoples');

        return view('frontend.pages.package-show', compact('destination', 'package', 'availableSeats'));
    }

    public function faq()
    {
        return view('frontend.pages.faq');
    }

    public function contact()
    {
        return view('frontend.pages.contact');
    }

    public function contactStore(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string',
            'email' => 'required|email',
            'phone' => 'required|string',
            'subject' => 'required|string',
            'message' => 'required|string',
            'check' => 'nullable|boolean',
        ]);

        $toEmail = setting('email');

        if ($toEmail) {
            Mail::to($toEmail)->send(new ContactUs($data));
        }

        if ($request->check) {
            Newsletter::subscribe($data['email']);
        }

        return back()->with('success', 'Your message has been sent successfully.');
    }

    public function blog(Blog $blog)
    {
        return view('frontend.pages.blog', compact('blog'));
    }

    public function subscribe(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        Newsletter::subscribe($request->email);

        return back()->with('success', 'You have been subscribed successfully.');
    }
}
